'use client';

import { useState, useEffect, useRef } from 'react';
import { InputText } from 'primereact/inputtext';
import { Card } from 'primereact/card';
import { Image } from 'primereact/image';
// 评分图标
import { Fire } from '@icon-park/react';
// 分页
import { Paginator, type PaginatorPageChangeEvent } from 'primereact/paginator';
// 页面跳转
import { useParams } from 'next/navigation'
import { useRouter } from 'next/navigation';
// 消息提醒
import { Toast } from 'primereact/toast';
// 接口传输
import axios from 'axios';
// 防抖函数
import { debounce } from 'lodash';
// 样式
import './resource-community-list.scss';

// 单元社区信息
interface Community {
    communityId: number;
    communityName: string;
    hot: number; // 热度
    threadNumber: number; // 帖子数量
    description: string; // 简介
    communityPicture: string; // 头像
}

// 资源社区列表
interface CommunityList {
    total: number;
    records: Community[];
}


// 资源社区界面
export default function ResourceCommunityPage() {
    // 获取URL参数
    const params = useParams<{ category: string }>()
    const category = decodeURIComponent(params.category); // 防止中文路径乱码
    const router = useRouter();

    // 社区列表数据
    const [communities, setCommunities] = useState<Community[]>([]);
    const [totalCommunities, setTotalCommunities] = useState<number>(0);
    // 消息提醒
    const toast = useRef<Toast>(null);
    //搜索框
    const [searchValue, setSearchValue] = useState('');
    const debouncedSearch = useRef(
        debounce((value: string) => {
            setSearchValue(value);
        }, 600)
    ).current;

    //分页
    const [first, setFirst] = useState(0);
    const [rows, setRows] = useState(5);
    const onPageChange = (event: PaginatorPageChangeEvent) => {
        setFirst(event.first);
        setRows(event.rows);
    };


    // 获取社区列表
    useEffect(() => {
        fetchCommunities();
    }, [category, first, rows, searchValue]);

    const fetchCommunities = async () => {
        try {
            const pageNumber = first / rows + 1;
            const type = category === 'all' ? '全部' : category;
            console.log("当前页" + pageNumber + "size" + rows + "type" + type + "searchValue" + searchValue);
            const response = await axios.get<CommunityList>(
                process.env.PUBLIC_URL + `/community`, {
                params: { type, pageNumber, rows, searchValue }
            }
            );
            console.log('获取社区列表:', response.data.records);
            setTotalCommunities(response.data.total);
            setCommunities(response.data.records);
        } catch (err) {
            console.error('获取社区失败', err);
            toast.current?.show({ severity: 'error', summary: 'error', detail: '获取社区失败' });
        }
    };


    return (
        <div className="resource-community-list-container">
            <Toast ref={toast}></Toast>
            <div className="resource-communities-header">
                <h1 className="title">
                    {category === 'all' ? '全部社区' : `${category}社区`}
                </h1>
            </div>

            {/* 社区搜索栏 */}
            <div className="searchBar">
                <i className="pi pi-search" />
                <InputText type="search" className="search-helper" placeholder="搜索你感兴趣的社区" onChange={(e) => { const target = e.target as HTMLInputElement; debouncedSearch(target.value); }} />
            </div>


            {/* 社区列表 */}
            <div className="resource-communities-list">
                {communities.map((community) => (
                    <Card key={community.communityId} className="resource-communities-list-card" onClick={() => router.push(`/community/community-detail/${community.communityId}`)}>
                        <Image alt="avatar" src={ community.communityPicture} className="community-avatar" width="250" height="140" />
                        <div className="community-header">
                            <div className="community-content">
                                <h3>{community.communityName}</h3>
                                <p className="community-introduction">{community.description}</p>
                            </div>
                            <div className="community-states">
                                <div className="state-item">
                                    <Fire theme="outline" size="16" fill="#FF8D1A" />
                                    <span>热度: {community.hot}</span>
                                </div>
                                <div className="state-item">
                                    <i className="pi pi-book" />
                                    <span>帖子数: {community.threadNumber}</span>
                                </div>
                            </div>
                        </div>
                    </Card>
                ))}
                {totalCommunities > 5 && <Paginator className="Paginator" first={first} rows={rows} totalRecords={totalCommunities} rowsPerPageOptions={[5, 10]} onPageChange={onPageChange} />}
            </div>
        </div>
    );
}